package myposhbot;

import cz.cuni.pogamut.MessageObjects.Mover;
import cz.cuni.pogamut.MessageObjects.Player;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.logging.Logger;

import cz.cuni.astar.AStar;
import cz.cuni.astar.AStarGoal;
import cz.cuni.astar.AStarMap;
import cz.cuni.astar.AStarResult;
import cz.cuni.pogamut.Client.AgentBody;
import cz.cuni.pogamut.Client.AgentMemory;
import cz.cuni.pogamut.Client.GameMap;
import cz.cuni.pogamut.Client.GameMapAStarGoal;
import cz.cuni.pogamut.Client.GameMapAStarMap;
import cz.cuni.pogamut.Client.GameMapSettings;
import cz.cuni.pogamut.Client.PathTypes;
import cz.cuni.pogamut.Client.RcvMsgEvent;
import cz.cuni.pogamut.communication.CommunicationState;
import cz.cuni.pogamut.MessageObjects.Health;
import cz.cuni.pogamut.MessageObjects.Item;
import cz.cuni.pogamut.MessageObjects.MessageObject;
import cz.cuni.pogamut.MessageObjects.NavPoint;
import cz.cuni.pogamut.MessageObjects.NeighNav;
import cz.cuni.pogamut.MessageObjects.Path;
import cz.cuni.pogamut.MessageObjects.Triple;
import cz.cuni.pogamut.MessageObjects.MessageType;
import java.util.List;

/**
 * auxiliary class for sorting Items according to distance from the bot
 * used in nearestHealths for instance
 * 
 * @author Horatko
 *
 */
class ItemDistance implements Comparable {
	public Item item = null;
	public int distance = 0;
	
	public ItemDistance(Item item, int distance) {
		this.item = item;
		this.distance = distance;
	}

	public int compareTo(Object arg0) {
		return this.distance - ((ItemDistance) arg0).distance;
	}

}
/**
 * GameMap provides simple navigation information
 * e.g. nearest navigation point, path to specified object
 * <p>
 * it also integrates AStar algorithm and use it to answer request for get path. 
 * this option is valid only for paths to NavPoints and Items, paths to players
 * are solved using Gamebot API
 * 
 * 
 * @author Horatko
 */

// opet bych dal hodne za to kdyby to vsechno byl jen interface
public class MyGameMap extends GameMap {
	/** logger for logging any messages for debugging etc. */
	protected Logger platformLog;
	/** poiter to body to use things like getPath(), knownObjects, addListener etc. */
	private AgentBody body = null;
	/** pointer to the memory - used by getPathAStar() 
	 * and other memory related topics - like getPathToClosestMedKit etc.
	 * */
	private AgentMemory memory = null; 
	
	/** Variables for runAroundItemsInTheMap */
	/** next item bot has to go */
	private Item nextItem = null;
	/** next item index in itemsToRunAround */
	private int nextItemIndex = 0;
	
        public MyPathManager pathManager = null;
        
	/** END OF VARIABLES FOR runAroundItemsInTheMap */
	
    public MyGameMap(Logger logger) {
        super(logger);
        this.platformLog = logger;
    }

    public MyGameMap(Logger logger, AgentBody body) {
        super(logger, body);
        this.platformLog = logger;
        this.body = body;
        this.pathManager = new MyPathManager(this);
        this.body.addTypedRcvMsgListener(this, MessageType.PATH);
        this.body.addRcvMsgListener(this, EnumSet.of(MessageType.SPAWN)); // for automatic memory restart, after agent's death

    }

    public MyGameMap(Logger logger, AgentBody body, AgentMemory memory) {
        super(logger, body, memory);
        this.platformLog = logger;
        this.body = body;
        this.body.addTypedRcvMsgListener(this, MessageType.PATH); // register listener for PATH message

        this.pathManager = new MyPathManager(this);
        this.memory = memory;
        this.body.addRcvMsgListener(this, EnumSet.of(MessageType.SPAWN)); // for automatic memory restart, after agent's death

    }

	/** 
	 * very basic method for getting healths of strenght greater or equal to strenght
	 * numberOfHealths specifies how many do you want
	 * <p>
	 * for more complex and more efficient way of doing this use something else
	 * think about map preprocessing and more efficient methods of search through the space
	 * (than using AStar numerous times)
	 * <p>
	 * note, it uses restricted AStar - restricted number of iterations, so it can find nothing!
	 * 
	 * @return list of healths ordered according to distance from the player (using built-in AStar to figure that out)
	 */
    @Override
	public ArrayList<Item> nearestHealth(int strength, int numberOfHealths) {
		ArrayList<Health> candidates = this.memory.getKnownHealths();
		ArrayList<Health> processedCandidates = new ArrayList<Health>();
		// preprocessing of candidates according to specified strenght
		for (Health health : candidates)
			if (health.strength >= strength)
				processedCandidates.add(health);
		// variables for storing whatever is necessary during the select sort
		ArrayList<ItemDistance> res = new ArrayList<ItemDistance>();
		AStarResult aStarResult = null;
		for (Health health : processedCandidates) {
			aStarResult = this.getPathAStar(health.navPoint);
			if (aStarResult.success)
				res.add(new ItemDistance(health, aStarResult.getDistanceToGoal()));
		}
		// sort the ItemDistances according to distance
		Collections.sort(res);
		// put them to the result - array of items that will be returned, pick only first numberOfHealths ones
		ArrayList<Item> result = new ArrayList<Item>();
		for (int i = 0; i < res.size() && i < numberOfHealths; i++)
			result.add(res.get(i).item);
		return result;
    }

	/** 
	 * very basic method for getting restricted number of closest items of specified type
	 * <p>
	 * for more complex and more efficient way of doing this use something else
	 * think about map preprocessing and more efficient methods of search through the space
	 * (than using AStar numerous times)
	 * <p>
	 * note, it uses restricted AStar - restricted number of iterations, so it can find nothing!
	 * 
	 * @return list of healths ordered according to distance from the player (using built-in AStar to figure that out)
	 */
    @Override
	public ArrayList<Item> nearestItems(MessageType type, int numberOfItems) {
		ArrayList<Item> candidates = this.memory.getKnownItemsOfType(type);
		// variables for storing whatever is necessary during the select sort
		ArrayList<ItemDistance> res = new ArrayList<ItemDistance>();
		AStarResult aStarResult = null;
		for (Item item : candidates) {
			aStarResult = this.getPathAStar(item.navPoint);
			if (aStarResult.success)
				res.add(new ItemDistance(item, aStarResult.getDistanceToGoal()));
		}
		// sort the ItemDistances according to distance
		Collections.sort(res);
		// put them to the result - array of items that will be returned, pick only first numberOfItems ones
		ArrayList<Item> result = new ArrayList<Item>();
		for (int i = 0; i < res.size() && i < numberOfItems; i++)
			result.add(res.get(i).item);
		return result;
	}
	
	/**
	 * Finds nearest navPoint - presume that the agent doesn't see any, so it uses
	 * this unefficient method
	 * <p>
	 * needs Triple from as a parameter
	 * @param from - most often agent location
	 * @return NavPoint - nearest NavPoint
	 */
    @Override
	public NavPoint nearestNavPoint(Triple from) {
		return nearestNavPoint(from, 0);
	}
	
	/**
	 * Finds nearest navPoint - presume that the agent doesn't see any, so it uses
	 * this unefficient method
	 * <p>
	 * counts only those in bigger distance than specified
	 * <p>
	 * needs Triple from as a parameter
	 * @param from - usualy agent location
	 * @return NavPoint - nearest NavPoint
	 */
    @Override
	public NavPoint nearestNavPoint(Triple from, int minDistance) {
		ArrayList<MessageObject> candidates = this.body.knownObjects.getObjectsOfType(MessageType.NAV_POINT);
		double distance = Double.MAX_VALUE;
		double currentDistance = distance;
		NavPoint closest = null;
		NavPoint temp = null;
		for (MessageObject candidate : candidates) {
			temp = (NavPoint) candidate;
			if (temp.location == null)
				continue;
			currentDistance = Triple.distanceInSpace(temp.location, from);
			if (distance > currentDistance && minDistance < currentDistance) {
				distance = currentDistance;
				closest = temp;
			}
		}
		return closest;
	}
	/**
         * @param location - location bot is heading to, comparing to
         * @return true if bot is about 10 unreal units from the provided location
         */
    @Override
        public boolean atLocation(Triple location) {
            return (Triple.distanceInSpace(location, this.memory.getAgentLocation()) < 10);
        }
	/**
         * 
         * @param location - location bot is heading to, comparing to
         * @param epsilon - a distance in UT units from which bot starts to be at the location
         * @return true if bot is about "epsilon" unreal units from the provided location
         */
    @Override
        public boolean atLocation(Triple location, int epsilon) {
            return (Triple.distanceInSpace(location, this.memory.getAgentLocation()) < epsilon);
        }
	/**
     * This method sends GETPATH query to GameBots - you have to provide PathID under
     * which the path will be returned by GB and stored in GameMap instance.<br>
     * Use getPathOfID(pathID) for retrieval.
     * 
     * 
     * @param toWhat - target (NavPoint) of the request
     */	
    @Override
	public void sendGetPathToNavPoint(NavPoint toWhat) {
		this.body.getPath(toWhat.getLocation(), toWhat.ID);
	}
	
	/**
     * This method sends GETPATH query to GameBots - you have to provide PathID under
     * which the path will be returned by GB and stored in GameMap instance.<br>
     * Use getPathOfID(pathID) for retrieval.
     * 
     * 
     * @param ID - id which is ment to be returned with query, toWhat - location (triple)
     */	
    @Override
	public void sendGetPathToLocation(int ID, Triple toWhat) {
		this.body.getPath(toWhat, ID);
	}
	
	/**
	 * Method for obtaining the path returned by GB due to the GETPATH request
	 * sent through sendGetPath(navPoint, pathID). <br>
	 * Note that it takes some time to resolve the GETPATH query.
	 * @param ID
	 * @return Path or null if ID does not exists
	 */
    @Override
	public ArrayList<NavPoint> getPathOfID(int ID) {
            return pathManager.collectedPaths.get(ID);
        }
	
	/**
	 * Returns path from your current position to the 'toWhat' or null if
	 * path doesn't exist.
	 * @return Collection of instances of NavPoint
	 */
    @Override
	public AStarResult getPathAStar(NavPoint toWhat, int maxNumOfIterations) {
		if (toWhat == null) return null;
		AStarGoal goal = new GameMapAStarGoal(toWhat);
		AStarMap map = new GameMapAStarMap();
		
		NavPoint start = new NavPoint();
		start.location = memory.getAgentLocation();
		start.neighbours = new ArrayList<NeighNav>();
		ArrayList<NavPoint> nvs = memory.seeAllNavPoints();
		NeighNav nn;
		for (int i = 0; i < nvs.size(); ++i){
			if (nvs.get(i).reachable) {
				nn = new NeighNav();
				nn.neighbour = nvs.get(i);
				start.neighbours.add(nn);
			}
		}
		assert goal != null;
		assert map != null;
		assert start != null;
		return AStar.aStar(goal, map, start, maxNumOfIterations);		
	}	
	
	/**
	 * auxiliary A* method <br>
	 * converts AStarResult to ArrayList of NavPoints which is then used as path in runAlongItemsInTheMap
	 * @param result - AStarResult
	 * @return list of navigation points which define path to object
	 */
    @Override
	public ArrayList<NavPoint> getNavPointsAStar(AStarResult result) {
		if (result.success){
			ArrayList path = result.getPath();
			ArrayList<NavPoint> navPoints = new ArrayList<NavPoint>();
			for (Object pathNode : path){
				if (pathNode instanceof NavPoint) navPoints.add((NavPoint)pathNode);
				else
				if (pathNode instanceof NeighNav) navPoints.add(((NeighNav)pathNode).neighbour);
				else
					platformLog.severe("Wrong object in the A* path ("+pathNode.toString()+").");
			}
			return navPoints;
		} else {
			return null;
		}
	}
	
	/**
	 * Returns path from your current position to the 'toWhat' or null if
	 * path doesn't exist.
	 * @return Collection of instances of NavPoint
	 */
    @Override
	public AStarResult getPathAStar(NavPoint toWhat) {
		AStarResult result = getPathAStar(toWhat, -1);		
		if (!result.success)
                    platformLog.info("PATH A*: failed unable to get path A Star");
		return result;		
	}
	
	/**
	 * add message to collected paths (adds) <br>
	 * FIFO fashion	
	 * <ul>
	 * <li>		 adds at the beginning
	 * <li>		 removes from the end 
	 * </ul>
	 */
    @Override
	public void receiveMessage(RcvMsgEvent e) {
		if (e.getCommunicationState() != CommunicationState.BOT_RUNNING)
			return;
		switch (e.getMessage().type) {
                case PATH: 
                        this.pathManager.newGBPathReceived((Path) e.getMessage());
			break;
		case SPAWN:
			this.restartMap();
			break;
		default:
			this.platformLog.warning("Unexpected message!!!");
		}
	}        
        
    @Override
        public void initializeGetPath() {
            pathManager.pathRequestSent = false;
        }
        
        /**
         * obtain path to specified location keeps returning null until path is received
         * don't forget to initialize it - initializeGetPath should be called prior this method
         * @param location
         * @return null if it has not received path from Gamebots yet, path if it does
         */
    @Override
        public ArrayList<NavPoint> getPathToLocation(Triple location) {
            return pathManager.getPathToLocation(location);
        }
        
        /** 
         * check if bot is stucked - if it is attempting too many times to reach some location and is not progressing
         *
         */
    @Override
        public boolean stuckCheck() {
                // check whether agent is stucked - yet in the rough testing
                boolean stucked = this.antiStuckCheck(this.pathManager.getCurrentNavPointOfPath().location);
                boolean attempts = false;
                ++pathManager.walking;
                // it has 15 attempts to reach the navPoint, if it fails, return false
                if (pathManager.walking > GameMapSettings.maxWalkingAttempts) {
                        attempts = true;
                }
                if (attempts && stucked) {                    
                    attempts = false;
                    platformLog.fine("REPLAN because bot reached max ATTEMPTS and was STUCKED");
                    return false;
                } 
                return true;
        }
        
	/**
	 * method to follow smoothly desired path - list of navigation points obtained from game map
	 * can handle movers - wait on it until top and leave
         * note that movers in the way are in this fashion:
         *  - LiftExit -> LiftCenter -> LiftExit
         * so as agent is supposed to go up when he is supposed to go to LiftCenter (therefore is around first LiftExit)
         * he waits for mover to be about his level and then goes on it, wait to level with second LiftExit and move on
	 * 
	 * @return boolean value - true if there are no problems, false if agent reaches the end or is unable to reach next navPoint
	 */
    @Override
	public boolean runAlongPath() {
		if (pathManager.pathToRunAlong == null) {
			platformLog.warning("procedure called when the variables were not properly initialized!");
			return false;
		}
                // end of the path
                if (pathManager.getCurrentNavPointOfPath() == null) {
                    platformLog.fine("runAlongPath - PATH END : " + pathManager.pathToRunAlong);
                    return false;
                }
		
                NavPoint nav1 = pathManager.getCurrentNavPointOfPath();
                
                // bot is to close to the next navPoint, switch to another
                // don't switch to another when you are at the end ... or more precisely switch to it at different distance from the navigation point
                if ((pathManager.getNextNavPointOfPath() == null) && 
                    (pathManager.nextNavPointIndex < pathManager.pathToRunAlong.size()) && 
                    (Triple.distanceInSpace(this.memory.getAgentLocation(), nav1.location) < GameMapSettings.lastNavigationPointOfPathPrecision)) {
                    platformLog.info("Last nav point of the path approached.");
                    pathManager.walking = 0;
                    ++pathManager.nextNavPointIndex;
                    return true;
                } else {
                    if ((Triple.distanceInSpace(this.memory.getAgentLocation(), nav1.location) < GameMapSettings.switchingDistance)) {
                        platformLog.info("Switching to another navigation point. Last point: " + nav1.UnrealID);
			pathManager.walking = 0;
			++pathManager.nextNavPointIndex;
                        return true;
                    }
		}
                // navpoint is null - don't know why, but even such a things happen
                if (nav1 == null || nav1.UnrealID == null) {
                    ++pathManager.nextNavPointIndex;
                    return true;
                }
                // bot is in the moving along the center of the lift center - switch to another nav point (lift exit)
                if ((nav1.UnrealID.indexOf("LiftCenter") != -1) && 
                    (Triple.distanceInPlane(this.memory.getAgentLocation(), nav1.location) < GameMapSettings.switchingDistance)) {
                        platformLog.info("Switching to another navigation point. Last point: " + nav1.UnrealID);
			pathManager.walking = 0;
			++pathManager.nextNavPointIndex;            
                        return true;
                }    
		// bot is still on its way, call moveAlong if he still has more than 1 navpoint on the remaining path
		// use runTo if there is only one
		if (pathManager.getNextNavPointOfPath() != null) {
                    NavPoint nav2 = pathManager.getNextNavPointOfPath();
                    // TODO: now it is only for the way up, not tested for movers which goes down as I don't know suitable map
                    // catch possible null pointer exception
                    try {
                        // agent is going to enter the lift (LiftCenter) and he is going through it (LiftExit) and 
                        // agent is not on the top of the lift
                        if ((nav1.UnrealID.indexOf("LiftCenter") != -1) && 
                            (nav2.UnrealID.indexOf("LiftExit") != -1) &&
                            ((this.memory.getAgentLocation().z - nav1.location.z) < 50)) {
                            // if agent sees mover - go on, else fail the run along - no possibility to move on the lift without mover
                            if (!this.memory.getSeeAnyMover())
                                return false;
                            else {
                                Mover mov = memory.getSeeMover();
                                if ((mov.location.z - this.memory.getAgentLocation().z) > GameMapSettings.levelForEnteringLift) {// mover is up - wait for it
                                    platformLog.fine("Waiting for MOVER to come DOWN");
                                    ++pathManager.walking;
                                    if (pathManager.walking > GameMapSettings.maxWalkingAttempts) {
                                        pathManager.walking = 0;
                                        return false;
                                    }
                                    return true;
                                } else {                        // enter the mover
                                    platformLog.fine("ENTERING the Mover");
                                    this.body.runToNavPoint(nav1); 
                                    pathManager.isUpOnTheLift = true;
                                    return this.stuckCheck();
                                }   
                            }
                        } else {
                            // let's go from the mover - so wait until agent got up and leave
                            if ((pathManager.isUpOnTheLift) && (nav1.UnrealID.indexOf("LiftExit") != 1)) {
                                if (!this.memory.getSeeAnyMover())
                                    return false;
                                else {
                                    // agent's z (height) coordinate is about the same level with lift exit - leave mover
                                    if ((nav1.location.z - this.memory.getAgentLocation().z) < GameMapSettings.levelForLeavingLift) {// mover is up - leave
                                        platformLog.fine("LEAVING MOVER!");
                                        pathManager.isUpOnTheLift = false;
                                        this.body.runToNavPoint(nav1);
                                        return this.stuckCheck();
                                    } else {
                                        // still going up on the mover - turn to the nav point at the top and wait 
                                        body.turnToLocation(nav1.location);
                                        // special stuck check - as agent is on the lift, he will be moving everytime, so it checks
                                        // only the lenght of his stay 
                                        ++pathManager.walking;
                                        if (pathManager.walking > GameMapSettings.maxWalkingAttempts) {
                                            pathManager.walking = 0;
                                            return false;
                                        }
                                        platformLog.fine("GOING UP " + (nav1.location.z - this.memory.getAgentLocation().z));
                                        return true;
                                    }
                                }
                            } else {
                                // normal movement along 2 points
                                this.body.moveAlongNavPoints(1, nav1, nav2);
                                return this.stuckCheck();
                            }
                        }
                    } catch (NullPointerException e) {
                        platformLog.warning("Null pointer exception in run along path " + e.getMessage());
                    }
                } else {
                    if (pathManager.nextNavPointIndex < pathManager.pathToRunAlong.size()) {
                        this.body.runToNavPoint(nav1);
                        return this.stuckCheck();
                    } 
                }   
                // fail - nowhere to go
                return false;
        }
	/**
	 * run around items in the map is a complex method which covers behavior which
	 * runs around the map according to specified list of items, it works in the following
	 * fashion:
	 * <ul>
	 * <li>	1) nextItem == null 	=> 	shuffle itemsToRunAround, choose first one
	 * <li>	2) path == null 		=> 	obtain path
	 * <li>	3) run along the path
 	 * <li>	4) if the run along fails or if bot reaches the item, switch to next on
 	 * </ul>
 	 * 
 	 * @param itemsToRunAround - array of items - like all weapons in the map
 	 * @param AStar - switch = to use or not to use built-in AStar - both work about the same
	 */
    @Override
	public void runAroundItemsInTheMap(List<? extends Item> itemsToRunAround, boolean AStar) {
                if (itemsToRunAround.size() == 0) {
                    platformLog.warning("no items to run around!!!");
                    return;
                }
                // 1) shuffle itemsToRunAround, choose first nextItem and restart the rest of variables
                if (this.nextItem == null) {
                        platformLog.fine("TOTAL RESTART OF RUN AROUND ITEMS IN MAP");
			Collections.shuffle(itemsToRunAround);
			this.nextItem = itemsToRunAround.get(0);
    			this.nextItemIndex = 0;
			return;
		}
		if (this.nextItem == null) // just to prevent possible null pointer exception
			return;
                
		// 2) obtain path - could end up in ethernal loop if the path doesn't exists
                if (pathManager.checkPath(PathTypes.ITEM, nextItem)) {
                    if (this.pathManager.pathToRunAlong.size() < 1){
                        platformLog.info("too short path to run along to next - restart");
                        this.nextItem = null;
                        return;
                    }
                    // 3) run along the path to chosen item - initialization is made separately in the previous section
                    if (this.runAlongPath()) {
                        return;
                    }                     
                } else {
                    pathManager.preparePath(PathTypes.ITEM, nextItem, AStar); 
                    return;
                }
                
                // 4) the bot reached the end of path, pick next item, initialize variables and continue
		if (++nextItemIndex < itemsToRunAround.size()) {
                    this.nextItem = itemsToRunAround.get(nextItemIndex);
                    platformLog.info("SWITCHING - running to another item in the list. \nNext item is: " + this.nextItem.cls);
		} else
                    // bot reached the last item, choose another one
                    this.nextItem = null;
	}
	
	/**
	 * method which restarts game map instance - restart necessary variables
	 *
	 */
    @Override
	public void restartMap() {
		this.nextItem = null;
		this.nextItemIndex = 0;
                this.pathManager = new MyPathManager(this);
	}
	
	
	int tempCounter = 0;
	/**
	 * check two things:
	 * 1) agent's velocity - size of vector greater than 50 UT units
	 * 2) agent is heading to the location
	 * 		- distance from the location is greater about 50 UT units than distance
	 * 		  from object after adding agent's velocity vector to location vector
	 * 
	 * - if this method fails - delete the edge in the graph for A*
	 * 
	 * @param goal - target location
	 * @return false if agent is not stucked (running and not heading to something too high)
	 */
    @Override
	public boolean antiStuckCheck(Triple goal) {
		Triple velocity = this.memory.getAgentVelocity();
		Triple location = this.memory.getAgentLocation();
		Triple nextPosition = Triple.add(velocity, location);
		double distanceDifference = Triple.vectorSize(Triple.subtract(goal, location));
		distanceDifference -= Triple.vectorSize(Triple.subtract(goal, nextPosition));
		if (distanceDifference < GameMapSettings.distanceDifference) {
			platformLog.finest("DIFFERENCE " + tempCounter);
			tempCounter += 1;
			return true;
		} 
		if (Triple.vectorSize(velocity) < GameMapSettings.minimalVelocity) {
			platformLog.finest("VELO " + Triple.vectorSize(velocity));
			return true;
		}
		if ((goal.z - location.z) > GameMapSettings.heightLevel) {
			platformLog.finest("HIGH HIGH HIGH HIGH ");
			return true;
		}
		// this should result in erasing proper edge as it is not easy to get there
		return false;
	}
        
        
        /**
	 * safely navigates agent to the location of choosen item (could be anywhere).
         * moreover you can set the distance at which agent will stop when reaching last point of the path
         * (by setting GameMapSettings.lastNavigationPointOfPathPrecision) it should be I think at least 5
         * as I think it is not guaranteed that bots moves precisely.
         * <br>
         * Note that it may have problems when running to something really close to agent, like when you order him to run to something
         * just on his side (lets say relatively 2 meters from him)
         * 
         * @param location - location to run to
         * @return false if there is some problem on the path - check platform log for more details
         */
    @Override
	public boolean safeRunToLocation(Triple location) {
            // if bot is close to the item there is a chance that he will not receive path!!! so he runs towards the location
            if ((Triple.distanceInSpace(this.memory.getAgentLocation(), location) < 500)) {
                    body.runToLocation(location); // experimental!!!
                    return true;
                }
            if (!pathManager.checkPath(PathTypes.LOCATION, location)) {
                pathManager.preparePath(PathTypes.LOCATION, location, false);
                
                return true;
            } else
                return runAlongPath();
	}

        /**
	 * safely navigates agent to the player (could be anywhere)
	 */
    @Override
	public boolean safeRunToPlayer(Player plr) {
            if (!pathManager.checkPath(PathTypes.PLAYER, plr)) {
                pathManager.preparePath(PathTypes.PLAYER, plr, false);
                return true;
            } else
                return runAlongPath();
	}
        
    @Override
        public void initializeRunAlongPath(ArrayList<NavPoint> path) {
            pathManager.initializeRunAlongPathManually(path);
        }
    
// strafing code
    
    public boolean StrafeAlongPath(Triple LookAt) {
            if (pathManager.pathToRunAlong == null) {
                    platformLog.warning("procedure called when the variables were not properly initialized!");
                    return false;
            }
            // end of the path
            if (pathManager.getCurrentNavPointOfPath() == null) {
                platformLog.fine("StrafeAlongPath - PATH END : " + pathManager.pathToRunAlong);
                return false;
            }

            NavPoint nav1 = pathManager.getCurrentNavPointOfPath();

            // bot is to close to the next navPoint, switch to another
            // don't switch to another when you are at the end ... or more precisely switch to it at different distance from the navigation point
            if ((pathManager.getNextNavPointOfPath() == null) && 
                (pathManager.nextNavPointIndex < pathManager.pathToRunAlong.size()) && 
                (Triple.distanceInSpace(this.memory.getAgentLocation(), nav1.location) < GameMapSettings.lastNavigationPointOfPathPrecision)) {
                platformLog.info("Last nav point of the path approached.");
                pathManager.walking = 0;
                ++pathManager.nextNavPointIndex;
                return true;
            } else {
                if ((Triple.distanceInSpace(this.memory.getAgentLocation(), nav1.location) < GameMapSettings.switchingDistance)) {
                    platformLog.info("Switching to another navigation point. Last point: " + nav1.UnrealID);
                    pathManager.walking = 0;
                    ++pathManager.nextNavPointIndex;
                    return true;
                }
            }
            // navpoint is null - don't know why, but even such a things happen
            if (nav1 == null || nav1.UnrealID == null) {
                ++pathManager.nextNavPointIndex;
                return true;
            }
            // bot is in the moving along the center of the lift center - switch to another nav point (lift exit)
            if ((nav1.UnrealID.indexOf("LiftCenter") != -1) && 
                (Triple.distanceInPlane(this.memory.getAgentLocation(), nav1.location) < GameMapSettings.switchingDistance)) {
                    platformLog.info("Switching to another navigation point. Last point: " + nav1.UnrealID);
                    pathManager.walking = 0;
                    ++pathManager.nextNavPointIndex;            
                    return true;
            }    
            // bot is still on its way, call moveAlong if he still has more than 1 navpoint on the remaining path
            // use runTo if there is only one
            if (pathManager.getNextNavPointOfPath() != null) {
                NavPoint nav2 = pathManager.getNextNavPointOfPath();
                // TODO: now it is only for the way up, not tested for movers which goes down as I don't know suitable map
                // catch possible null pointer exception
                try {
                    // agent is going to enter the lift (LiftCenter) and he is going through it (LiftExit) and 
                    // agent is not on the top of the lift
                    if ((nav1.UnrealID.indexOf("LiftCenter") != -1) && 
                        (nav2.UnrealID.indexOf("LiftExit") != -1) &&
                        ((this.memory.getAgentLocation().z - nav1.location.z) < 50)) {
                        // if agent sees mover - go on, else fail the run along - no possibility to move on the lift without mover
                        if (!this.memory.getSeeAnyMover())
                            return false;
                        else {
                            Mover mov = memory.getSeeMover();
                            if ((mov.location.z - this.memory.getAgentLocation().z) > GameMapSettings.levelForEnteringLift) {// mover is up - wait for it
                                platformLog.fine("Waiting for MOVER to come DOWN");
                                ++pathManager.walking;
                                if (pathManager.walking > GameMapSettings.maxWalkingAttempts) {
                                    pathManager.walking = 0;
                                    return false;
                                }
                                return true;
                            } else {                        // enter the mover
                                platformLog.fine("ENTERING the Mover");
                                this.body.runToNavPoint(nav1); 
                                pathManager.isUpOnTheLift = true;
                                return this.stuckCheck();
                            }   
                        }
                    } else {
                        // let's go from the mover - so wait until agent got up and leave
                        if ((pathManager.isUpOnTheLift) && (nav1.UnrealID.indexOf("LiftExit") != 1)) {
                            if (!this.memory.getSeeAnyMover())
                                return false;
                            else {
                                // agent's z (height) coordinate is about the same level with lift exit - leave mover
                                if ((nav1.location.z - this.memory.getAgentLocation().z) < GameMapSettings.levelForLeavingLift) {// mover is up - leave
                                    platformLog.fine("LEAVING MOVER!");
                                    pathManager.isUpOnTheLift = false;
                                    this.body.strafeToLocation(nav1.location, LookAt);
                                    return this.stuckCheck();
                                } else {
                                    // still going up on the mover - turn to the nav point at the top and wait 
                                    body.turnToLocation(nav1.location);
                                    // special stuck check - as agent is on the lift, he will be moving everytime, so it checks
                                    // only the lenght of his stay 
                                    ++pathManager.walking;
                                    if (pathManager.walking > GameMapSettings.maxWalkingAttempts) {
                                        pathManager.walking = 0;
                                        return false;
                                    }
                                    platformLog.fine("GOING UP " + (nav1.location.z - this.memory.getAgentLocation().z));
                                    return true;
                                }
                            }
                        } else {
                            // normal movement along 2 points
                            this.body.strafeToLocation(nav1.location, LookAt);
                            platformLog.severe("normal movement");
                            return this.stuckCheck();
                        }
                    }
                } catch (NullPointerException e) {
                    platformLog.warning("Null pointer exception in run along path " + e.getMessage());
                }
            } else {
                if (pathManager.nextNavPointIndex < pathManager.pathToRunAlong.size()) {
                    this.body.strafeToLocation(nav1.location, LookAt);
                    platformLog.severe("to druhy");
                    return this.stuckCheck();
                } 
            }   
            // fail - nowhere to go
            return false;
    }
    
    /**
     * safely navigates agent to the location of choosen item (could be anywhere).
     * moreover you can set the distance at which agent will stop when reaching last point of the path
     * (by setting GameMapSettings.lastNavigationPointOfPathPrecision) it should be I think at least 5
     * as I think it is not guaranteed that bots moves precisely.
     * <br>
     * Note that it may have problems when running to something really close to agent, like when you order him to run to something
     * just on his side (lets say relatively 2 meters from him)
     * 
     * @param location - location to run to
     * @return false if there is some problem on the path - check platform log for more details
     */
    public boolean safeStrafeToLocation(Triple location, Triple LookAt) {
        // if bot is close to the item there is a chance that he will not receive path!!! so he runs towards the location
        //userlog.severe("safeSTRAFE");
        if ((Triple.distanceInSpace(this.memory.getAgentLocation(), location) < 500)) {
                //userlog.severe("STRAFE TO LOCATION");
                body.strafeToLocation(location, LookAt); // experimental!!!
                return true;
            }
        if (!pathManager.checkPath(PathTypes.LOCATION, location)) {
            //userlog.severe("CHECK PATH");
            //userlog.severe("location: " + location.toString() + "LookAt: " + LookAt.toString());
            pathManager.preparePath(PathTypes.LOCATION, location, true);
            return true;
        } else {
            platformLog.severe("STRAFE ALONG PATH");
            return StrafeAlongPath(LookAt);
        }
    }

    /**
     * safely navigates agent to the player (could be anywhere)
     */
    public boolean safeStrafeToPlayer(Player plr, Triple LookAt) {
        if (!pathManager.checkPath(PathTypes.PLAYER, plr)) {
            pathManager.preparePath(PathTypes.PLAYER, plr, false);
            return true;
        } else
            return StrafeAlongPath(LookAt);
    }    
}
